<?php

/**
 * Module widget_meteo pour ZwiiCMS
 *
 * Affiche un widget météo avec redirection vers une page dédiée
 * fait appel au module ZwiiCMS forecat_openmeteo 
 * API : https://open-meteo.com/
 * 
 * @package    ZwiiCMS
 * @subpackage Module
 * @author     Michel Tuboeuf
 * @license    CC Attribution-NonCommercial-NoDerivatives 4.0 International
 * @link       https://zwiicms.fr/
 * @version    1.1
 */
class widget_meteo extends common
{

    const VERSION = '1.1';
    const REALNAME = 'Widget Météo';
    const DELETE = true;
    const UPDATE = '0.0';

    public static $actions = [
        'index' => self::ROLE_VISITOR,
        'config' => self::ROLE_EDITOR
    ];

    public static $datas = [];

    /**
     * Fonctions utilitaires pour la météo
     */

    /** Récupère la prévision pour l'heure actuelle */
    public static function getCurrentHourForecast($forecast)
    {
        if (empty($forecast)) {
            return null;
        }

        $currentHour = date('H:00');
        $currentTimestamp = time();

        foreach ($forecast as $hour) {
            if (isset($hour['timestamp']) && abs($hour['timestamp'] - $currentTimestamp) < 3600) {
                return $hour;
            }
            if (isset($hour['time_fmt']) && $hour['time_fmt'] === $currentHour) {
                return $hour;
            }
        }
        return null;
    }

    /** Convertit les degrés en direction du vent */
    public static function wind_direction($deg)
    {
        if (!is_numeric($deg)) {
            return '';
        }

        static $dirs = ['N', 'NNE', 'NE', 'ENE', 'E', 'ESE', 'SE', 'SSE', 'S', 'SSO', 'SO', 'OSO', 'O', 'ONO', 'NO', 'NNO'];
        return $dirs[(int)(($deg + 11.25) / 22.5) % 16];
    }

    /** Retourne l'emoji correspondant à la condition météo */
    public static function get_weather_emoji($condition)
    {
        if (empty($condition)) {
            return '🌦️';
        }

        static $emojiMap = [
            'soleil' => '☀️',
            'clair' => '☀️',
            'ensoleillé' => '☀️',
            'nuage' => '☁️',
            'nuageux' => '☁️',
            'pluie' => '🌧️',
            'pluvieux' => '🌧️',
            'orage' => '⛈️',
            'orages' => '⛈️',
            'neige' => '❄️',
            'neigeux' => '❄️',
            'brouillard' => '🌫️',
            'brume' => '🌫️',
        ];
        $condition = strtolower($condition);
        foreach ($emojiMap as $key => $emoji) {
            if (strpos($condition, $key) !== false) {
                return $emoji;
            }
        }
        return '🌦️';
    }

    /** Affiche le message d'absence de données */
    public static function renderNoData()
    {
        return '
        <div class="weather-widget">
            <div class="no-data">
                <div style="font-size: 48px; margin-bottom: 8px;">☁️</div>
                <p>Données météo indisponibles.</p>
                <small>Vérifiez votre connexion ou réessayez plus tard.</small>
            </div>
        </div>
        ';
    }

    /** Formate l'heure de mise à jour */
    public static function formatUpdateTime($timestamp)
    {
        if ($timestamp) {
            $dt = new DateTime('@' . $timestamp);
            $dt->setTimezone(new DateTimeZone('Europe/Paris'));
            return $dt->format('d/m/Y H:i');
        }
        return 'inconnue';
    }

    /** Prépare les données météo pour l'affichage */
    public static function prepareWeatherData($weather, $lastUpdate)
    {
        if (!$weather || empty($weather['current'])) {
            return null;
        }

        $current = $weather['current'];
        $forecast = $weather['hourly_forecast'] ?? [];

        // Remplacer les données current par celles de l'heure actuelle des prévisions
        $currentHourData = self::getCurrentHourForecast($forecast);
        if ($currentHourData) {
            $current = array_merge($current, $currentHourData);
        }

        return [
            'current' => $current,
            'forecast' => $forecast,
            'city' => $current['city_name'] ?? 'Ville inconnue',
            'tempC' => round($current['temp'], 1),
            'windKph' => round($current['wind_speed'], 1),
            'windDir' => isset($current['wind_deg']) ? self::wind_direction($current['wind_deg']) : '',
            'humidity' => round($current['humidity']),
            'condition' => $current['condition'],
            'icon' => !empty($current['condition_icon']) ? $current['condition_icon'] : self::get_weather_emoji($current['condition']),
            'sunrise' => isset($current['sunrise']) ? date('H:i', strtotime($current['sunrise'])) : '',
            'sunset' => isset($current['sunset']) ? date('H:i', strtotime($current['sunset'])) : '',
            'updateTime' => self::formatUpdateTime($lastUpdate),
            'windGust' => isset($current['wind_gust']) ? round($current['wind_gust'], 1) : null
        ];
    }

    /** Affichage du widget */
    public function index()
    {
        // Charger la configuration
        $config = $this->getData(['module', $this->getUrl(0)]);

        // Chemin vers le fichier de données météo
        $weatherFile = __DIR__ . '/../../site/data/weather_openmeteo/weather.json';
        $weather = null;
        $lastUpdate = null;

        // Chargement optimisé des données
        if (file_exists($weatherFile)) {
            $content = file_get_contents($weatherFile);
            if ($content !== false) {
                $data = json_decode($content, true);
                if (json_last_error() === JSON_ERROR_NONE && !empty($data['cache']['data'])) {
                    $weather = $data['cache']['data'];
                    $lastUpdate = $data['cache']['last_updated'] ?? null;
                }
            }
        }

        // Préparer les données pour la vue
        self::$datas = [
            'weather' => $weather,
            'lastUpdate' => $lastUpdate,
            'redirectUrl' => $config['redirectUrl'] ?? '',
            'moduleId' => $this->getUrl(0),
            'weatherData' => self::prepareWeatherData($weather, $lastUpdate)
        ];

        // Utiliser le système de vues ZwiiCMS
        $this->addOutput([
            'title' => 'Widget Météo',
            'view' => 'index',
            'showBarEditButton' => true,
            'showPageContent' => true,
        ]);
    }

    /** Configuration du module */
    public function config()
    {
        // Soumission du formulaire
        if ($this->isPost()) {
            $input = $this->getInput('widget_meteoRedirectUrl');
            $this->setData([
                'module',
                $this->getUrl(0),
                [
                    'redirectUrl' => $input,
                    'userId' => $this->getUser('id'),
                    'updateTime' => time()
                ]
            ]);

            // Redirection avec le système ZwiiCMS
            $this->addOutput([
                'redirect' => helper::baseUrl() . $this->getUrl(0) . '/config',
                'notification' => 'Configuration enregistrée'
            ]);
            return;
        }

        // Charger la configuration actuelle
        $config = $this->getData(['module', $this->getUrl(0)]);

        // Préparer les données pour la vue
        self::$datas = [
            'redirectUrl' => $config['redirectUrl'] ?? '',
            'moduleId' => $this->getUrl(0)
        ];

        // Utiliser le système de vues ZwiiCMS
        $this->addOutput([
            'title' => 'Configuration Widget Météo',
            'view' => 'config',
            'showBarEditButton' => true,
            'showPageContent' => false,  // Désactive l'affichage du contenu parent
            'template' => 'empty',       // Utilise un template vide si possible
        ]);
    }
}
