<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/**
 * Module widget_meteo pour ZwiiCMS
 *
 * Affiche un widget météo avec redirection vers une page dédiée
 * Utilise l'API Open-Meteo pour des données réelles
 * 
 * @package    ZwiiCMS
 * @subpackage Module
 * @author     Michel Tuboeuf
 * @version    1.4
 */

class widget_meteo extends common
{
    // Constantes de classe : version, nom, et autorisation de suppression
    const VERSION = '1.4';
    const REALNAME = 'Widget Météo';
    const DELETE = true;

    // Définition des actions et des rôles associés
    public static $actions = [
        'index' => self::ROLE_VISITOR, // Accessible aux visiteurs
        'config' => self::ROLE_EDITOR  // Accessible aux éditeurs
    ];

    // Tableau pour stocker les données du module
    public static $datas = [];

    // Configuration par défaut du widget
    private static $defaultConfig = [
        'city' => 'Paris',
        'latitude' => '48.8566',
        'longitude' => '2.3522',
        'redirectUrl' => '',
        'dataSource' => 'api'
    ];

    // Durée de cache en secondes (15 minutes)
    private $cacheDuration = 900;

    // Constructeur de la classe
    public function __construct()
    {
        parent::__construct();
    }

    // --------------------------
    // FONCTIONS UTILITAIRES
    // --------------------------

    // Récupère la prévision pour l'heure actuelle
    public static function getCurrentHourForecast($forecast)
    {
        // Logique pour trouver la prévision correspondant à l'heure actuelle
        if (empty($forecast)) return null;

        $currentHour = date('H:00');
        $currentTimestamp = time();

        foreach ($forecast as $hour) {
            if (!empty($hour['timestamp']) && abs($hour['timestamp'] - $currentTimestamp) < 3600) {
                return $hour;
            }
            if (!empty($hour['time_fmt']) && $hour['time_fmt'] === $currentHour) {
                return $hour;
            }
        }
        return null;
    }

    // Convertit les degrés en direction cardinale (N, NE, E, etc.)
    public static function wind_direction($deg)
    {
        if (!is_numeric($deg)) return '';
        $dirs = ['N', 'NNE', 'NE', 'ENE', 'E', 'ESE', 'SE', 'SSE', 'S', 'SSO', 'SO', 'OSO', 'O', 'ONO', 'NO', 'NNO'];
        return $dirs[(int)(($deg + 11.25) / 22.5) % 16];
    }

    // Retourne un emoji en fonction de la condition météo     
    public static function get_weather_emoji($condition)
    {
        if (empty($condition)) return '🌦️';
        $emojiMap = [
            'soleil' => '☀️',
            'clair' => '☀️',
            'ensoleillé' => '☀️',
            'nuage' => '☁️',
            'nuageux' => '☁️',
            'pluie' => '🌧️',
            'pluvieux' => '🌧️',
            'orage' => '⛈️',
            'orages' => '⛈️',
            'neige' => '❄️',
            'neigeux' => '❄️',
            'brouillard' => '🌫️',
            'brume' => '🌫️'
        ];
        $condition = strtolower($condition);
        foreach ($emojiMap as $key => $emoji) {
            if (strpos($condition, $key) !== false) return $emoji;
        }
        return '🌦️';
    }

    // Affiche un message si les données météo sont indisponibles
    public static function renderNoData()
    {
        return '<div class="weather-widget"><div class="no-data">
            <div style="font-size:48px;margin-bottom:8px;">☁️</div>
            <p>Données météo indisponibles.</p>
            <small>Vérifiez votre connexion ou réessayez plus tard.</small>
        </div></div>';
    }

    // Formate le timestamp en date lisible
    public static function formatUpdateTime($timestamp)
    {
        return $timestamp ? (new DateTime("@$timestamp", new DateTimeZone('Europe/Paris')))->format('d/m/Y H:i') : 'inconnue';
    }

    // Prépare les données météo pour l'affichage
    public static function prepareWeatherData($weather, $lastUpdate)
    {
        if (empty($weather['current'])) return null;

        $current = $weather['current'];
        $forecast = $weather['hourly_forecast'] ?? [];

        $hasRealData = isset($current['temp']) && $current['temp'] !== null;
        if ($hasRealData) {
            $currentHourData = self::getCurrentHourForecast($forecast);
            if ($currentHourData) $current = array_merge($current, $currentHourData);
        }

        return [
            'current' => $current,
            'forecast' => $forecast,
            'city' => $current['city_name'] ?? 'Ville inconnue',
            'tempC' => $current['temp'] !== null ? round($current['temp'], 1) : '--',
            'windKph' => $current['wind_speed'] !== null ? round($current['wind_speed'], 1) : '--',
            'windDir' => isset($current['wind_deg']) ? self::wind_direction($current['wind_deg']) : '--',
            'humidity' => $current['humidity'] !== null ? round($current['humidity']) : '--',
            'condition' => $current['condition'] ?? 'Inconnu',
            'icon' => !empty($current['condition_icon']) ? $current['condition_icon'] : self::get_weather_emoji($current['condition'] ?? ''),
            'sunrise' => isset($current['sunrise']) ? date('H:i', strtotime($current['sunrise'])) : '--:--',
            'sunset' => isset($current['sunset']) ? date('H:i', strtotime($current['sunset'])) : '--:--',
            'updateTime' => self::formatUpdateTime($lastUpdate),
            'windGust' => isset($current['wind_gust']) ? round($current['wind_gust'], 1) : null,
            'hasRealData' => $hasRealData
        ];
    }

    // --------------------------
    // API & CACHE
    // --------------------------

    // Récupère les données météo depuis l'API Open-Meteo ou le cache
    private function fetchWeatherFromAPI($latitude, $longitude, $city)
    {
        $cacheFile = __DIR__ . '/../../site/data/widget_meteo/widget_meteo_cache.json';
        $cacheTTL = 900; // en secondes. 900 = 15 mn

        if (file_exists($cacheFile)) {
            $cache = json_decode(file_get_contents($cacheFile), true);
            if (
                is_array($cache)
                && strtolower($cache['data']['current']['city_name'] ?? '') === strtolower($city)
                && (time() - ($cache['last_updated'] ?? 0) < $cacheTTL)
            ) return $cache['data'];
        }

        $url = "https://api.open-meteo.com/v1/forecast?" . http_build_query([
            'latitude' => $latitude,
            'longitude' => $longitude,
            'current' => 'temperature_2m,relative_humidity_2m,apparent_temperature,weather_code,wind_speed_10m,wind_direction_10m,wind_gusts_10m',
            'hourly' => 'temperature_2m,weather_code,wind_speed_10m,wind_direction_10m',
            'daily' => 'sunrise,sunset',
            'timezone' => 'Europe/Paris',
            'forecast_days' => 2
        ]);

        $res = $this->httpGet($url);
        if (!$res['ok']) {
            error_log("Widget Météo - API Error: " . $res['error']);
            return $cache['data'] ?? null;
        }
        $data = json_decode($res['body'], true);
        if (!$data) {
            error_log("Widget Météo - API Invalid");
            return $cache['data'] ?? null;
        }

        $formatted = $this->formatAPIData($data, $city);

        if (!is_dir(dirname($cacheFile))) mkdir(dirname($cacheFile), 0755, true);
        file_put_contents($cacheFile, json_encode(['last_updated' => time(), 'data' => $formatted], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        return $formatted;
    }

    // Formate les données brutes de l'API pour une utilisation plus simple
    private function formatAPIData($apiData, $city)
    {
        $current = $apiData['current'] ?? [];
        $hourly = $apiData['hourly'] ?? [];
        $daily = $apiData['daily'] ?? [];

        $weatherCodes = [
            0 => 'Ciel dégagé',
            1 => 'Principalement dégagé',
            2 => 'Partiellement nuageux',
            3 => 'Nuageux',
            45 => 'Brouillard',
            48 => 'Brouillard givrant',
            51 => 'Bruine légère',
            53 => 'Bruine modérée',
            55 => 'Bruine dense',
            61 => 'Pluie légère',
            63 => 'Pluie modérée',
            65 => 'Pluie forte',
            80 => 'Averses légères',
            81 => 'Averses modérées',
            82 => 'Averses violentes',
            95 => 'Orage',
            96 => 'Orage avec grêle légère',
            99 => 'Orage avec grêle forte'
        ];

        $formattedCurrent = [
            'city_name' => $city,
            'temp' => $current['temperature_2m'] ?? null,
            'feels_like' => $current['apparent_temperature'] ?? null,
            'humidity' => $current['relative_humidity_2m'] ?? null,
            'wind_speed' => $current['wind_speed_10m'] ?? null,
            'wind_deg' => $current['wind_direction_10m'] ?? null,
            'wind_gust' => $current['wind_gusts_10m'] ?? null,
            'condition' => $weatherCodes[$current['weather_code'] ?? -1] ?? 'Inconnu',
            'condition_icon' => $this->getWeatherIconFromCode($current['weather_code'] ?? -1),
            'sunrise' => $daily['sunrise'][0] ?? '',
            'sunset' => $daily['sunset'][0] ?? ''
        ];

        $formattedHourly = [];
        foreach (($hourly['time'] ?? []) as $i => $time) {
            $ts = strtotime($time);
            $formattedHourly[] = [
                'timestamp' => $ts,
                'time_fmt' => date('H:i', $ts),
                'temp' => $hourly['temperature_2m'][$i] ?? null,
                'wind_speed' => $hourly['wind_speed_10m'][$i] ?? null,
                'wind_deg' => $hourly['wind_direction_10m'][$i] ?? null,
                'condition' => $weatherCodes[$hourly['weather_code'][$i] ?? -1] ?? 'Inconnu',
                'condition_icon' => $this->getWeatherIconFromCode($hourly['weather_code'][$i] ?? -1)
            ];
        }

        // On garde les 6 heures à partir de l'heure la plus proche
        usort($formattedHourly, fn($a, $b) => abs($a['timestamp'] - time()) <=> abs($b['timestamp'] - time()));
        $formattedHourly = array_slice($formattedHourly, 0, 6);
        usort($formattedHourly, fn($a, $b) => $a['timestamp'] <=> $b['timestamp']);

        return ['current' => $formattedCurrent, 'hourly_forecast' => array_slice($formattedHourly, 0, 6)];
    }

    // Retourne un emoji en fonction du code météo de l'API
    private function getWeatherIconFromCode($code)
    {
        $map = [
            0 => '☀️',
            1 => '🌤️',
            2 => '⛅',
            3 => '☁️',
            45 => '🌫️',
            48 => '🌫️',
            51 => '🌦️',
            53 => '🌦️',
            55 => '🌧️',
            61 => '🌧️',
            63 => '🌧️',
            65 => '🌧️',
            80 => '🌦️',
            81 => '🌧️',
            82 => '⛈️',
            95 => '⛈️',
            96 => '⛈️',
            99 => '⛈️'
        ];
        return $map[$code] ?? '🌦️';
    }

    // Effectue une requête HTTP GET
    private function httpGet($url)
    {
        // Utilisation de cURL pour récupérer les données de l'API
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_USERAGENT => 'ZwiiCMS-WidgetMeteo/' . self::VERSION,
            CURLOPT_SSL_VERIFYPEER => false
        ]);
        $body = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $err = curl_error($ch);
        curl_close($ch);
        return ['ok' => $body !== false && $code === 200, 'body' => $body, 'error' => $err];
    }

    // --------------------------
    // AFFICHAGE MODULE
    // --------------------------

    // Affiche le widget météo (page principale)
    public function index()
    {
        $configFile = __DIR__ . '/../../site/data/widget_meteo/widget_meteo.json';
        $config = self::$defaultConfig;

        // Charger la configuration si le fichier existe
        if (file_exists($configFile)) {
            $fileConfig = json_decode(file_get_contents($configFile), true);
            if (is_array($fileConfig)) {
                $config = array_merge($config, $fileConfig);
            }
        }

        $debugHtml = '<div style="border:2px solid red;padding:5px;margin-bottom:10px;">
        <strong>DEBUG Widget Météo</strong><br>dataSource utilisé : ' . htmlspecialchars($config['dataSource'] ?? '') . '<br>';

        if ($config['dataSource'] === 'file') {
            $weatherFile = __DIR__ . '/../../site/data/weather_openmeteo/weather.json';
            $debugHtml .= file_exists($weatherFile) ? "Fichier externe trouvé : $weatherFile<br>" : "Fichier externe INTRouvable : $weatherFile<br>";
        } else {
            $debugHtml .= "Utilisation de l'API Open-Meteo<br>";
        }
        $debugHtml .= '</div>';

        if ($config['dataSource'] === 'file') {
            $weather = $lastUpdate = null;
            if (file_exists($weatherFile)) {
                $data = json_decode(file_get_contents($weatherFile), true);
                if (!empty($data['cache']['data'])) {
                    $weather = $data['cache']['data'];
                    $lastUpdate = $data['cache']['last_updated'] ?? null;
                }
            }
        } else {
            $weather = $this->fetchWeatherFromAPI($config['latitude'], $config['longitude'], $config['city']);
            $lastUpdate = time();
        }

        self::$datas = [
            'weather' => $weather,
            'lastUpdate' => $lastUpdate,
            'redirectUrl' => $config['redirectUrl'] ?? '',
            'moduleId' => $this->getUrl(0),
            'weatherData' => self::prepareWeatherData($weather, $lastUpdate),
            'debugHtml' => $debugHtml
        ];

        $this->addOutput([
            'title' => 'Widget Météo',
            'view' => 'index',
            'showBarEditButton' => true,
            'showPageContent' => true
        ]);
    }

    // Gère la configuration du widget (page de configuration)
    public function config()
    {
        $configFile = __DIR__ . '/../../site/data/widget_meteo/widget_meteo.json';

        // Créer le dossier s'il n'existe pas
        if (!is_dir(dirname($configFile))) {
            mkdir(dirname($configFile), 0755, true);
        }

        if ($this->isPost()) {
            $newConfig = [
                'city' => $_POST['city'] ?? self::$defaultConfig['city'],
                'latitude' => $_POST['latitude'] ?? self::$defaultConfig['latitude'],
                'longitude' => $_POST['longitude'] ?? self::$defaultConfig['longitude'],
                'redirectUrl' => $_POST['redirectUrl'] ?? self::$defaultConfig['redirectUrl'],
                'dataSource' => $_POST['dataSource'] ?? self::$defaultConfig['dataSource'],
                'userId' => $this->getUser('id'),
                'updateTime' => time()
            ];
            $res = file_put_contents($configFile, json_encode($newConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            $this->addOutput([
                'redirect' => helper::baseUrl() . $this->getUrl(0) . '/config',
                'notification' => $res ? 'Configuration enregistrée' : 'Erreur lors de l\'enregistrement',
                'state' => true,
            ]);
            return;
        }

        // Charger la configuration existante ou utiliser les valeurs par défaut
        $config = self::$defaultConfig;
        if (file_exists($configFile)) {
            $fileConfig = json_decode(file_get_contents($configFile), true);
            if (is_array($fileConfig)) {
                $config = array_merge($config, $fileConfig);
            }
        }

        self::$datas = [
            'city' => $config['city'],
            'latitude' => $config['latitude'],
            'longitude' => $config['longitude'],
            'redirectUrl' => $config['redirectUrl'],
            'dataSource' => $config['dataSource'],
            'moduleId' => $this->getUrl(0)
        ];

        $this->addOutput([
            'title' => 'Configuration Widget Météo',
            'view' => 'config',
        ]);
    }
}
